/**
 * This decision module intend to decide whether to offload before computing.
 * First, collecting environment factors, then building an approximate function
 * describing the cost value both on time and energy. Then, deciding which one is
 * benefit for user specified preference. Finally, do the computation.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <sys/stat.h>
#include "base.h"
#include "factor.h"
#include "cost_function.h"

extern float clamAV_scanner(const char *, struct FactorSet *);
extern float offload_to_cloud(const char *, struct FactorSet *);
static void compute(const int, struct FactorSet *, const char *);
static int make_decision(struct CostFunction *, float);

struct handler {
	char *unit;
	float (*function)(const char *, struct FactorSet *);
} func[3] = {
	{(char *)"CPU",   clamAV_scanner},
	{(char *)"GPU",   NULL},
	{(char *)"CLOUD", offload_to_cloud}
};

void decision_module(const char *param, const float alpha, const int specified)
{
	int errcode;
	int op_unit;
	struct FactorSet factor_set;
	struct CostFunction cost_func;
	FILE *fptr = NULL;
	int size;
	struct stat scanned_file_st;

	check_current_env();
		
	init_factor_set(&factor_set);

	if ((fptr = fopen(FACTOR_TABLE, "r")) == NULL) {
		/* Factor table not exist, first time execution */
		/* If folder "Docs" not exists, create it */
		if (access("Docs", 0) == -1) {
			if (mkdir("Docs", S_IRWXU | S_IRWXG | S_IRWXO) == -1) {
				system_error();
			}
		}
		create_factor_table(&factor_set);
		op_unit = CPU;
	}
	else {
		debug_print("INFO", "Collecting Factors\n");
		collect_factors(&factor_set, fptr);
		fclose(fptr);
		stat(param, &scanned_file_st);
		size = scanned_file_st.st_size;
		factor_set.N_input = size; /* scanned file size */
		factor_set.N_output = sizeof(int); 	/* only yes or no */

		debug_print("INFO", "Building Cost Functions\n");
		errcode = build_cost_functions(&factor_set, &cost_func, size);
		if (errcode != 0) {
			system_error();
		}

		debug_print("INFO", "Making Decision\n");
		op_unit = make_decision(&cost_func, alpha);

		/* If given specified target, then set the target for executing. */
		if (specified >= 0) {
			op_unit = specified;
		}
	}
	compute(op_unit, &factor_set, param);
	update_factor_table(&factor_set);
}


/*
 * @Function:		make_decision
 * @Input
 * 		cf:			cost functions
 * 		alpha:
 * @Description:	making decision according to cost functions
 * */
int make_decision(struct CostFunction *cf, float alpha)
{
	int i;
	float min_value = 0;
	float epsilon_cop_T, epsilon_cop_E;
	float epsilon_cld_T, epsilon_cld_E;
	float temp;
	int target = CPU;

	epsilon_cop_T = (cf->T_cop - cf->T_cpu)/cf->T_cpu;
	epsilon_cop_E = (cf->E_cop - cf->E_cpu)/cf->E_cpu;
	temp = alpha*epsilon_cop_T+(1-alpha)*epsilon_cop_E;
	if (min_value > temp) {
		min_value = temp;
		target = GPU;
	}
	printf("%f %f\n", epsilon_cop_T, epsilon_cop_E);

	epsilon_cld_T = (cf->T_cld - cf->T_cpu)/cf->T_cpu;
	epsilon_cld_E = (cf->E_cld - cf->E_cpu)/cf->E_cpu;
	temp = alpha*epsilon_cld_T+(1-alpha)*epsilon_cld_E;
	if (min_value > temp) {
		min_value = temp;
		target = CLOUD;
	}
	printf("%f %f\n", epsilon_cld_T, epsilon_cld_E);

	debug_print("INFO", "The minimum cost is use %s with %f us.\n", func[target].unit, min_value);

	FILE *fptr = fopen("Docs/dec-log", "a");
	fprintf(fptr, "%s\t", func[target].unit);
	fclose(fptr);

	return target;
}

/*
 * @Function:		compute
 * @Return:			none
 * @Input
 * 		op_unit:
 * 		fs:
 * 		times:
 * @Description:	
 */
static void compute(int op_unit, struct FactorSet *fs, const char *scanned_file)
{
	float exec_time = 0.0f;
	FILE *log_fp;

	exec_time = (func[op_unit].function)(scanned_file, fs);

	printf("%s spend: %f us\n", func[op_unit].unit, exec_time);
	/* Write execution time to log file */
	log_fp = fopen("Docs/exec-log", "a");
	fprintf(log_fp, " %f\n", exec_time);
	fclose(log_fp);
}
